﻿
/****************************************************************************/
/*Copyright (c) 2011, Florent DEVILLE.                                      */
/*All rights reserved.                                                      */
/*                                                                          */
/*Redistribution and use in source and binary forms, with or without        */
/*modification, are permitted provided that the following conditions        */
/*are met:                                                                  */
/*                                                                          */
/* - Redistributions of source code must retain the above copyright         */
/*notice, this list of conditions and the following disclaimer.             */
/* - Redistributions in binary form must reproduce the above                */
/*copyright notice, this list of conditions and the following               */
/*disclaimer in the documentation and/or other materials provided           */
/*with the distribution.                                                    */
/* - The names of its contributors cannot be used to endorse or promote     */
/*products derived from this software without specific prior written        */
/*permission.                                                               */
/* - The source code cannot be used for commercial purposes without         */
/*its contributors' permission.                                             */
/*                                                                          */
/*THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       */
/*"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT         */
/*LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS         */
/*FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE            */
/*COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,       */
/*INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,      */
/*BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;          */
/*LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER          */
/*CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT        */
/*LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN         */
/*ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE           */
/*POSSIBILITY OF SUCH DAMAGE.                                               */
/****************************************************************************/

using System.Collections.Generic;

using Microsoft.Xna.Framework;

namespace GE.Visualisation
{
    /// <summary>
    /// Store an animation composed of sprites from a single texture
    /// </summary>
    class Animation
    {
        /// <summary>
        /// Index of the texture containing the sprites
        /// </summary>
        int m_indexTexture;

        /// <summary>
        /// Time elapsed between each frame of the animation
        /// </summary>
	    int m_offsetTime;

        /// <summary>
        /// List of the sprite's index composing the animation
        /// </summary>
	    List<int> m_indexSprite;

        /// <summary>
        /// Liste of position offset for each animation frame
        /// </summary>
	    List<Vector2> m_offsetVector;

        /// <summary>
        /// Indicates if the animation uses the offset vectors
        /// </summary>
	    bool m_useOffset;

        /// <summary>
        /// Name of the animation
        /// </summary>
        string m_name;

        /// <summary>
        /// Return the index of the texture containing the animation' sprites
        /// </summary>
        public int indexTexture{get{return m_indexTexture;}}

        /// <summary>
        /// Return the elapse time between frames
        /// </summary>
        public int offsetTime{get{return m_offsetTime;}}

        /// <summary>
        /// Return the offset flag. True if the offset vectors are used or else false.
        /// </summary>
        public bool useOffset{get{return m_useOffset;}}

        /// <summary>
        /// Return the name of the animation
        /// </summary>
        public string Name { get { return m_name; } }

        /// <summary>
        /// Return the number of frame in the animation
        /// </summary>
        public int FrameCount { get { return m_indexSprite.Count; } }

        /// <summary>
        /// Constructeur
        /// </summary>
        /// <param name="name">Animation's name</param>
        /// <param name="indexTexture">Index of the texture containing the sprites</param>
        /// <param name="offsetTime">Time elapsed between frames</param>
        /// <param name="useOffset">Offset vector flag</param>
	    public Animation(string name, int indexTexture, int offsetTime, bool useOffset)
        {
            m_indexSprite = new List<int>();
            m_offsetVector = new List<Vector2>();

            m_name = name;
            m_indexTexture = indexTexture;
            m_offsetTime = offsetTime;
            m_useOffset = useOffset;
        }
	
	    /// <summary>
	    /// Add a sprite to the animation with a offset vector set to 0, 0
	    /// </summary>
	    /// <param name="indexSprite">the sprite's index to add to the animation</param>
	    public void addSprite(int indexSprite)
        {
            m_indexSprite.Add(indexSprite);
            m_offsetVector.Add(new Vector2(0));
        }

        /// <summary>
        /// Add a sprite to the animation and set its offset vector
        /// </summary>
        /// <param name="indexSprite">the sprite's index to add to the animation</param>
        /// <param name="offset">offset of the frame</param>
        public void addSprite(int indexSprite, Vector2 offset)
        {
            m_indexSprite.Add(indexSprite);
            m_offsetVector.Add(offset);
        }

	    /// <summary>
	    /// Return the sprite's index of the animation's last frame
	    /// </summary>
        public int idLastSprite { get { return m_indexSprite[m_indexSprite.Count-1]; } }

        /// <summary>
        /// Return the sprite's index of the animation's first frame
        /// </summary>
        public int idFirstSprite{get{return m_indexSprite[0];}}
	   
        /// <summary>
        /// Return the sprite's index of the n'th animation's frame
        /// </summary>
        /// <param name="index">frame index</param>
        /// <returns></returns>
	    public int getIndexSprite(int index){return m_indexSprite[index];}

        /// <summary>
        /// Return the offset of the n'th animation frame
        /// </summary>
        /// <param name="index">frame index</param>
        /// <returns></returns>
        public Vector2 getOffset(int index) { return m_offsetVector[index]; }

        
    }
}
